<?php

/**
 * Payment Helper
 *
 * @package     Cabme
 * @subpackage  Controller
 * @category    Payment
 * @author      SMR IT Solutions Team
 * @version     2.2.1
 * @link        https://smritsolutions.com
 */
namespace App\Http\Helper;

class PaymentHelper
{
	/**
	 * Braintree Payment
	 *
	 * @param Array $payment_data [payment_data includes currency, amount]
	 * @param String $[nonce] [nonce get it from braintree gateway]
	 * @return Boolean
	 */
	public function BraintreePayment($payment_data,$nonce)
	{
		$payment_currency = site_settings('payment_currency');
		$payment_amount = currencyConvert($payment_data['currency_code'],$payment_currency,$payment_data['amount']);
		try {
			$gateway = resolve('braintree');
			$result = $gateway->transaction()->sale([
				'amount' => $payment_amount,
				'paymentMethodNonce' => $nonce,
				'options' => [
					'submitForSettlement' => True
				]
			]);
		}
		catch (\Exception $e) {
			return arrayToObject([
				'status' => false,
				'status_message' => $e->getMessage(),
			]);
		}

		$return_data['status'] = $result->success;
		$return_data['is_two_step'] = false;
		if($result->success) {
			$return_data['transaction_id'] = $result->transaction->id;
		}
		else {
			$return_data['status_message'] = $result->message;
			logger($result->errors);
		}
		return arrayToObject($return_data);
	}

	/**
	 * Paypal Payment
	 *
	 * @param Array $payment_data [payment_data includes currency, amount]
	 * @param String $[nonce] [nonce get it from Braintree gateway]
	 * @return Boolean
	 */
	public function PaypalPayment($payment_data,$nonce)
	{
		$payment_currency = site_settings('payment_currency');
		$payment_amount = currencyConvert($payment_data['currency_code'],$payment_currency,$payment_data['amount']);
		try {
			/*
			Old PayPal Payment Method
			$gateway = resolve('paypal');
			$purchase_response = $gateway->fetchPurchase(['transactionReference' => $pay_key])->send();
			$transaction_id = $purchase_response->getTransactionReference() ?: '';*/
			$gateway = resolve('braintree_paypal');
			$result = $gateway->transaction()->sale([
				'amount' => $payment_amount,
				'paymentMethodNonce' => $nonce,
				'options' => [
					'submitForSettlement' => True
				]
			]);
		}
		catch (\Exception $exception) {
			return arrayToObject([
				'status' => false,
				'status_message' => $exception->getMessage(),
			]);
		}

		$return_data['status'] = $result->success;
		$return_data['is_two_step'] = false;
		if($result->success) {
			$return_data['transaction_id'] = $result->transaction->id;
		}
		else {
			$return_data['status_message'] = $result->message;
		}
		return arrayToObject($return_data);
	}

	/**
	 * Stripe Payment
	 *
	 * @param Array $payment_data [payment_data includes currency, amount]
	 * @param String $[intent_id] [intent_id get it from Stripe]
	 * @return Boolean
	 */
	public function StripePayment($payment_data,$intent_id = '')
	{
		$stripe_payment = resolve('App\Repositories\StripePayment');

		if($intent_id != '') {
			$payment_result = $stripe_payment->CompletePayment($intent_id);
		}
		else {
			$payment_result = $stripe_payment->createPaymentIntent($payment_data);
		}

		if($payment_result->status == 'success') {
			return arrayToObject([
				'status' => true,
				'is_two_step' => false,
				'transaction_id' => $payment_result->transaction_id,
			]);
		}
		else if($payment_result->is_two_step) {
			return arrayToObject([
				'status' => true,
				'is_two_step' => true,
				'status_message' => $payment_result->status,
				'two_step_id' => $payment_result->intent_client_secret,
			]);
		}

		return arrayToObject([
			'status' => false,
			'status_message' => $payment_result->status_message,
		]);
	}
}