<?php

/**
 * Payment Gateway Controller
 *
 * @package     Ridein
 * @subpackage  Controller
 * @category    Payment Gateway
 * @author      Source monster Team
 * @version     3.0.1
 * @link        https://sourcemonster.in
 */

namespace App\Http\Controllers\Admin;

use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use App\Models\PaymentGateway;

class PaymentGatewayController extends Controller
{
    /**
     * Load View and Update Payment Gateway Data
     *
     * @return redirect to payment_gateway
     */
    public function index(Request $request)
    {
        if ($request->isMethod('GET')) {

            $rawGateways = PaymentGateway::whereNotIn('site', ['Cash'])->get();

            $gatewayLists  = $rawGateways->groupBy('site')->map(function ($items, $site) {
                return [
                    'id' => $items->first()->id, // or some unique ID
                    'fields' => $items->map(function ($item) {
                        return [
                            'name' => $item->name,
                            'value' => $item->value,
                            'type' => $this->determineFieldType($item->name), // You can customize this logic
                        ];
                    }),
                ];
            });


            return view('admin.payment_gateway', compact('gatewayLists'));
        }
    }


    function determineFieldType(string $fieldName): string
    {
        $booleanFields    = ['is_enabled', 'mode'];  // fields you want as checkbox
        $checkboxFields  = ['payout_methods']; // if you want to keep any boolean separate

        if (in_array($fieldName,  $checkboxFields)) {
            return 'checkbox';
        }

        if (in_array($fieldName, $booleanFields)) {
            return 'boolean';
        }

        return 'text';
    }



    public function update(Request $request, $id)
    {

        $gateway = PaymentGateway::findOrFail($id);
        $site = $gateway->site;


        // Define validation rules dynamically
        $rules = [];

        switch (strtolower($site)) {
            case 'paypal':
                $rules = [
                    'fields.client' => 'required|string',
                    'fields.secret' => 'required|string',
                    'fields.paypal_id' => 'required|string',
                    'fields.mode' => 'required|in:sandbox,live',
                    'fields.access_token' => 'required|string',
                    'fields.is_enabled' => 'required|in:0,1',
                ];
                break;
            case 'stripe':
                $rules = [
                    'fields.publish' => 'required|string',
                    'fields.secret' => 'required|string',
                    'fields.is_enabled' => 'required|in:0,1',
                ];
                break;
            case 'braintree':
                $rules = [
                    'fields.public_key' => 'required|string',
                    'fields.merchant_id' => 'required|string',
                    'fields.private_key' => 'required|string',
                    'fields.tokenization_key' => 'required|string',
                    'fields.merchant_account_id' => 'required|string',
                    'fields.is_enabled' => 'required|in:0,1',
                ];
                break;
            case 'razorpay':
                $rules = [
                    'fields.key' => 'required|string',
                    'fields.secret' => 'required|string',
                    'fields.is_enabled' => 'required|in:0,1',
                    'fields.mode' => 'required|string',
                ];
                break;
            case 'paytm':
                $rules = [
                    'fields.merchant_id' => 'required|string',
                    'fields.key' => 'required|string',
                    'fields.website' => 'required|string',
                    'fields.channel' => 'required|string',
                    'fields.type' => 'required|string',
                    'fields.is_enabled' => 'required|in:0,1',
                ];
                break;
            case 'flutterwave':
                $rules = [
                    'fields.key' => 'required|string',
                    'fields.secret' => 'required|string',
                    'fields.hash' => 'required|string',
                    'fields.is_enabled' => 'required|in:0,1',
                ];
                break;
            case 'zaincash':
                $rules = [
                    'fields.merchant_id' => 'required|string',
                    'fields.merchant_secret' => 'required|string',
                    'fields.msisdn' => 'required|string',
                    'fields.is_enabled' => 'required|in:0,1',
                ];
                break;
            case 'mtn':
                $rules = [
                    'fields.collection_id' => 'required|string',
                    'fields.subscription_key' => 'required|string',
                    'fields.secret' => 'required|string',
                    'fields.is_enabled' => 'required|in:0,1',
                    'fields.collection_id_payout' => 'required|string',
                    'fields.subscription_key_payout' => 'required|string',
                    'fields.secret_payout' => 'required|string',
                ];
                break;
            case 'yo':
                $rules = [
                    'fields.username' => 'required|string',
                    'fields.password' => 'required|string',
                    'fields.is_enabled' => 'required|in:0,1',
                ];
                break;
            case 'redsys':
                $rules = [
                    'fields.merchant_code' => 'required|string',
                    'fields.key' => 'required|string',
                    'fields.terminal' => 'required|string',
                    'fields.url_notification' => 'required|url',
                    'fields.url_ok' => 'required|url',
                    'fields.url_ko' => 'required|url',
                    'fields.tradename' => 'required|string',
                    'fields.is_enabled' => 'required|in:0,1',
                ];
                break;
            case 'mpesa':
                $rules = [
                    'fields.mode' => 'required|string',
                    'fields.consumer_secret' => 'required|string',
                    'fields.consumer_key' => 'required|string',
                    'fields.business_shortcode' => 'required|string',
                    'fields.pass_key' => 'required|string',
                    'fields.is_enabled' => 'required|in:0,1',
                ];
                break;
            case 'zaincash':
                $rules = [
                    'fields.mode' => 'required|string',
                    'fields.merchant_id' => 'required|string',
                    'fields.merchant_secret' => 'required|string',
                    'fields.msisdn' => 'required|string',
                    'fields.is_enabled' => 'required|in:0,1',
                ];
                break;
            case 'paynow':
                $rules = [
                    'fields.integration_id' => 'required|string',
                    'fields.integration_key' => 'required|string',
                    'fields.is_enabled' => 'required|in:0,1',
                ];
                break;
            case 'paystack':
                $rules = [
                    'fields.secret_key' => 'required|string',
                    'fields.public_key' => 'required|string',
                    'fields.is_enabled' => 'required|in:0,1',
                ];
                break;
            case 'common':
                $rules = [
                    'fields.payout_methods' => 'required|array',
                ];
                break;
            default:

                break;
        }


        $messages = [
            'required' => 'The :attribute field is required.',
            'in'       => 'The selected :attribute is invalid.',
            'string'   => 'The :attribute must be a valid string.',
            'url'      => 'The :attribute must be a valid URL.',
        ];


        $attributes = [];
        $prefix = ucfirst($site); // e.g., "Paypal"

        foreach ($rules as $key => $rule) {
            $field = str_replace('fields.', '', $key);
            $readable = ucfirst(str_replace('_', ' ', $field)); // e.g., "client_id" -> "Client Id"
            $attributes[$key] = "$prefix $readable";            // e.g., "Paypal Client Id"
        }


        try {
            $validated = $request->validate($rules, $messages, $attributes);

            // Ensure only one gateway is enabled
            if (isset($validated['fields']['is_enabled']) && $validated['fields']['is_enabled'] === '1') {
                // Disable all other 'is_enabled' flags
                PaymentGateway::where('name', 'is_enabled')
                    ->where('site', '!=', $site)
                    ->where('site', '!=', 'Cash')
                    ->update(['value' => '0']);
            }


            // Save updated fields
            foreach ($validated['fields'] as $name => $value) {

                // Only convert 'payout_methods' array to comma-separated string
                if ($name === 'payout_methods' && is_array($value)) {
                    $value = implode(',', $value);
                }

                PaymentGateway::where('site', $site)
                    ->where('name', $name)
                    ->update(['value' => $value]);
            }


            flashMessage('success', 'Updated Successfully');

            return redirect('admin/payment_gateway');
        } catch (\Illuminate\Validation\ValidationException $e) {

            return back()->withErrors($e->validator)->withInput();
        }
    }
}
