<?php

/**
 * Paystack Payment Service
 *
 * @package     Ridein
 * @subpackage  Services\Payments
 * @category    Paystack
 * @author      Sourcemonster Team
 * @version     3.8.0
 * @link        https://sourcemonster.in
 */

namespace App\Services\PaymentMethods;


use App\Contracts\PaymentInterface;
use App\Models\Transaction;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Config;
use Unicodeveloper\Paystack\Facades\Paystack;

class PaystackPayment implements PaymentInterface
{


    /**
     * Intialize Stripe with Secret key
     *
     */
    public function __construct()
    {
        Config::set('paystack.publicKey', payment_gateway('public_key', 'Paystack'));
        Config::set('paystack.secretKey', payment_gateway('secret_key', 'Paystack'));
        $mode = payment_gateway('mode', 'Paystack');
        Config::set('paystack.paymentUrl', $mode === 'live' ? 'https://api.paystack.co' : 'https://api.paystack.co');
    }

    public function makePayment($payment_details, $indent_id = '')
    {
        try {

            $reference = Paystack::genTranxRef();
            $data = [
                'reference' => $reference,
                'email' => $payment_details['user_email'],
                'name' => $payment_details['user_name'],
                'amount' => $payment_details['amount'] * 100,
                'currency' => $payment_details['currency_code'] ?? 'NGN',
                'metadata' => json_encode([
                    'pay_for' => ucfirst($payment_details['pay_for']),
                    'user_id' => $payment_details['user_id'],
                    'trip_id' => $payment_details['trip_id'] ?? 0,
                    'user_phone' => $payment_details['user_phone']
                ])
            ];

            $authorizationUrl =  Paystack::getAuthorizationUrl($data);
            if (!isset($authorizationUrl->url) || !$authorizationUrl->url) {
                return arrayToObject(array(
                    'status'         => false,
                    'status_message' => 'Payment initialization failed',
                ));
            }

            $transaction = new Transaction();
            $transaction->payment_method = 'Paystack';
            $transaction->payment_purpose =  ucfirst($payment_details['pay_for']);
            $transaction->user_id = $payment_details['user_id'];
            $transaction->trip_id = $payment_details['trip_id'];
            $transaction->amount = $payment_details['amount'];
            $transaction->reference =   $reference;
            $transaction->status = 'pending';
            $transaction->currency = 'ZAR';
            $transaction->tel =  $payment_details['user_phone'];
            $transaction->save();


            $return_data['status'] = true;
            $return_data['status_message'] = 'payment successful';
            $return_data['payment_link'] = $authorizationUrl->url;

            return arrayToObject($return_data);
        } catch (\Exception $e) {
            Log::error('Paystack Payment Error: ' . $e->getMessage());

            return arrayToObject(array(
                'status'         => false,
                'status_message' => $e->getMessage(),
            ));
        }
    }


    public function paymentVerify($requestData)
    {
        try {
            $paymentDetails = Paystack::getPaymentData();

            if (!$paymentDetails['status']) {
                // Payment was successful
                return arrayToObject(array(
                    'status'         => false,
                    'status_message' => 'Payment verified failed',
                ));
            }
            $paystackData = $paymentDetails['data'];

            // Check if the payment was successful
            $return_data['status'] = true;
            $return_data['transaction_id'] =  $paystackData['reference'];
            $return_data['status_message']  = "Payment Successfully!";

            return arrayToObject($return_data);
        } catch (\Exception $e) {
            Log::error('Paystack Payment Verification Error: ' . $e->getMessage());
            return arrayToObject(array(
                'status'         => false,
                'status_message' => $e->getMessage(),
            ));
        }
    }


    public function handleWebHook($requestData, $signature)
    {

        $computedSignature = hash_hmac('sha512', $requestData->getContent(), config('paystack.secretKey'));

        if (!hash_equals($computedSignature, $signature)) {
            return arrayToObject(array(
                'status'         => false,
                'status_message' => 'Invalid webhook signature',
            ));
        }

        $event = $requestData->input('event');
        $data = $requestData->input('data');

        if (!isset($event) || $event !== 'charge.success') {
            Log::error('Paystack Webhook Error: Invalid event');
            $getTransaction = Transaction::where('reference', $data['reference'])->first();
            if ($getTransaction) {
                $getTransaction->status = 'failed';
                $getTransaction->save();
            }


            return arrayToObject(array(
                'status'         => false,
                'status_message' => 'Invalid webhook event',
            ));
        }

        $getTransaction = Transaction::where('reference', $data['reference'])->where('status', 'pending')->first();
        if ($getTransaction) {
            $getTransaction->status = 'completed';
            $getTransaction->save();
        }

        // Handle the webhook logic here
        return arrayToObject(array(
            'status'         => true,
            'status_message' => 'Webhook handled successfully',
        ));
    }
}
